/*
 * Decompiled with CFR 0.152.
 */
package de.maxhenkel.audioplayer.microhttp;

import de.maxhenkel.audioplayer.microhttp.ByteTokenizer;
import de.maxhenkel.audioplayer.microhttp.Cancellable;
import de.maxhenkel.audioplayer.microhttp.CloseUtils;
import de.maxhenkel.audioplayer.microhttp.Handler;
import de.maxhenkel.audioplayer.microhttp.Header;
import de.maxhenkel.audioplayer.microhttp.LogEntry;
import de.maxhenkel.audioplayer.microhttp.Logger;
import de.maxhenkel.audioplayer.microhttp.Options;
import de.maxhenkel.audioplayer.microhttp.Request;
import de.maxhenkel.audioplayer.microhttp.RequestParser;
import de.maxhenkel.audioplayer.microhttp.Response;
import de.maxhenkel.audioplayer.microhttp.Scheduler;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

class ConnectionEventLoop {
    private final Options options;
    private final Logger logger;
    private final Handler handler;
    private final AtomicLong connectionCounter;
    private final AtomicBoolean stop;
    private final Scheduler timeoutQueue;
    private final Queue<Runnable> taskQueue;
    private final ByteBuffer buffer;
    private final Selector selector;
    private final Thread thread;

    ConnectionEventLoop(Options options, Logger logger, Handler handler, AtomicLong connectionCounter, AtomicBoolean stop) throws IOException {
        this.options = options;
        this.logger = logger;
        this.handler = handler;
        this.connectionCounter = connectionCounter;
        this.stop = stop;
        this.timeoutQueue = new Scheduler();
        this.taskQueue = new ConcurrentLinkedQueue<Runnable>();
        this.buffer = ByteBuffer.allocateDirect(options.readBufferSize());
        this.selector = Selector.open();
        this.thread = new Thread(this::run, "connection-event-loop");
    }

    int numConnections() {
        return this.selector.keys().size();
    }

    void start() {
        this.thread.start();
    }

    void join() throws InterruptedException {
        this.thread.join();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void run() {
        try {
            this.doStart();
        }
        catch (IOException e) {
            if (this.logger.enabled()) {
                this.logger.log(e, new LogEntry("event", "sub_event_loop_terminate"));
            }
            this.stop.set(true);
        }
        finally {
            for (SelectionKey selKey : this.selector.keys()) {
                Object attachment = selKey.attachment();
                if (!(attachment instanceof Connection)) continue;
                Connection connection = (Connection)attachment;
                connection.failSafeClose();
            }
            CloseUtils.closeQuietly(this.selector);
        }
    }

    private void doStart() throws IOException {
        while (!this.stop.get()) {
            Runnable task;
            this.selector.select(this.options.resolution().toMillis());
            Set<SelectionKey> selectedKeys = this.selector.selectedKeys();
            Iterator<SelectionKey> it = selectedKeys.iterator();
            while (it.hasNext()) {
                SelectionKey selKey = it.next();
                if (selKey.isReadable()) {
                    ((Connection)selKey.attachment()).onReadable();
                } else if (selKey.isWritable()) {
                    ((Connection)selKey.attachment()).onWritable();
                }
                it.remove();
            }
            this.timeoutQueue.expired().forEach(Runnable::run);
            while ((task = this.taskQueue.poll()) != null) {
                task.run();
            }
        }
    }

    void register(SocketChannel socketChannel) {
        this.taskQueue.add(() -> {
            try {
                this.doRegister(socketChannel);
            }
            catch (IOException e) {
                this.logger.log(e, new LogEntry("event", "register_error"));
                CloseUtils.closeQuietly(socketChannel);
            }
        });
        this.selector.wakeup();
    }

    private void doRegister(SocketChannel socketChannel) throws IOException {
        socketChannel.configureBlocking(false);
        SelectionKey selectionKey = socketChannel.register(this.selector, 1);
        Connection connection = new Connection(socketChannel, selectionKey);
        selectionKey.attach(connection);
        if (this.logger.enabled()) {
            this.logger.log(new LogEntry("event", "accept"), new LogEntry("remote_address", socketChannel.getRemoteAddress().toString()), new LogEntry("id", connection.id));
        }
    }

    private class Connection {
        static final String HTTP_1_0 = "HTTP/1.0";
        static final String HTTP_1_1 = "HTTP/1.1";
        static final String HEADER_CONNECTION = "Connection";
        static final String HEADER_CONTENT_LENGTH = "Content-Length";
        static final String KEEP_ALIVE = "Keep-Alive";
        final SocketChannel socketChannel;
        final SelectionKey selectionKey;
        final ByteTokenizer byteTokenizer;
        final String id;
        RequestParser requestParser;
        ByteBuffer writeBuffer;
        Cancellable requestTimeoutTask;
        boolean httpOneDotZero;
        boolean keepAlive;

        private Connection(SocketChannel socketChannel, SelectionKey selectionKey) throws IOException {
            this.socketChannel = socketChannel;
            this.selectionKey = selectionKey;
            this.byteTokenizer = new ByteTokenizer();
            this.id = Long.toString(ConnectionEventLoop.this.connectionCounter.getAndIncrement());
            this.requestParser = new RequestParser(this.byteTokenizer);
            this.requestTimeoutTask = ConnectionEventLoop.this.timeoutQueue.schedule(this::onRequestTimeout, ConnectionEventLoop.this.options.requestTimeout());
        }

        private void onRequestTimeout() {
            if (ConnectionEventLoop.this.logger.enabled()) {
                ConnectionEventLoop.this.logger.log(new LogEntry("event", "request_timeout"), new LogEntry("id", this.id));
            }
            this.failSafeClose();
        }

        private void onReadable() {
            try {
                this.doOnReadable();
            }
            catch (IOException | RuntimeException e) {
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(e, new LogEntry("event", "read_error"), new LogEntry("id", this.id));
                }
                this.failSafeClose();
            }
        }

        private void doOnReadable() throws IOException {
            ConnectionEventLoop.this.buffer.clear();
            int numBytes = this.socketChannel.read(ConnectionEventLoop.this.buffer);
            if (numBytes < 0) {
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(new LogEntry("event", "read_close"), new LogEntry("id", this.id));
                }
                this.failSafeClose();
                return;
            }
            ConnectionEventLoop.this.buffer.flip();
            this.byteTokenizer.add(ConnectionEventLoop.this.buffer);
            if (ConnectionEventLoop.this.logger.enabled()) {
                ConnectionEventLoop.this.logger.log(new LogEntry("event", "read_bytes"), new LogEntry("id", this.id), new LogEntry("read_bytes", Integer.toString(numBytes)), new LogEntry("request_bytes", Integer.toString(this.byteTokenizer.remaining())));
            }
            if (this.requestParser.parse()) {
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(new LogEntry("event", "read_request"), new LogEntry("id", this.id), new LogEntry("request_bytes", Integer.toString(this.byteTokenizer.remaining())));
                }
                this.onParseRequest();
            } else if (this.byteTokenizer.size() > ConnectionEventLoop.this.options.maxRequestSize()) {
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(new LogEntry("event", "exceed_request_max_close"), new LogEntry("id", this.id), new LogEntry("request_size", Integer.toString(this.byteTokenizer.size())));
                }
                this.failSafeClose();
            }
        }

        private void onParseRequest() {
            if (this.selectionKey.interestOps() != 0) {
                this.selectionKey.interestOps(0);
            }
            if (this.requestTimeoutTask != null) {
                this.requestTimeoutTask.cancel();
                this.requestTimeoutTask = null;
            }
            Request request = this.requestParser.request();
            this.httpOneDotZero = request.version().equalsIgnoreCase(HTTP_1_0);
            this.keepAlive = request.hasHeader(HEADER_CONNECTION, KEEP_ALIVE);
            this.byteTokenizer.compact();
            this.requestParser = new RequestParser(this.byteTokenizer);
            ConnectionEventLoop.this.handler.handle(request, this::onResponse);
        }

        private void onResponse(Response response) {
            ConnectionEventLoop.this.taskQueue.add(() -> {
                try {
                    this.prepareToWriteResponse(response);
                }
                catch (IOException e) {
                    if (ConnectionEventLoop.this.logger.enabled()) {
                        ConnectionEventLoop.this.logger.log(e, new LogEntry("event", "response_ready_error"), new LogEntry("id", this.id));
                    }
                    this.failSafeClose();
                }
            });
            if (Thread.currentThread() != ConnectionEventLoop.this.thread) {
                ConnectionEventLoop.this.selector.wakeup();
            }
        }

        private void prepareToWriteResponse(Response response) throws IOException {
            String version = this.httpOneDotZero ? HTTP_1_0 : HTTP_1_1;
            ArrayList<Header> headers = new ArrayList<Header>();
            if (this.httpOneDotZero && this.keepAlive) {
                headers.add(new Header(HEADER_CONNECTION, KEEP_ALIVE));
            }
            if (!response.hasHeader(HEADER_CONTENT_LENGTH)) {
                headers.add(new Header(HEADER_CONTENT_LENGTH, Integer.toString(response.body().length)));
            }
            this.writeBuffer = ByteBuffer.wrap(response.serialize(version, headers));
            if (ConnectionEventLoop.this.logger.enabled()) {
                ConnectionEventLoop.this.logger.log(new LogEntry("event", "response_ready"), new LogEntry("id", this.id), new LogEntry("num_bytes", Integer.toString(this.writeBuffer.remaining())));
            }
            this.doOnWritable();
        }

        private void onWritable() {
            try {
                this.doOnWritable();
            }
            catch (IOException | RuntimeException e) {
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(e, new LogEntry("event", "write_error"), new LogEntry("id", this.id));
                }
                this.failSafeClose();
            }
        }

        private int doWrite() throws IOException {
            ConnectionEventLoop.this.buffer.clear();
            int amount = Math.min(ConnectionEventLoop.this.buffer.remaining(), this.writeBuffer.remaining());
            ConnectionEventLoop.this.buffer.put(this.writeBuffer.array(), this.writeBuffer.position(), amount);
            ConnectionEventLoop.this.buffer.flip();
            int written = this.socketChannel.write(ConnectionEventLoop.this.buffer);
            this.writeBuffer.position(this.writeBuffer.position() + written);
            return written;
        }

        private void doOnWritable() throws IOException {
            int numBytes = this.doWrite();
            if (!this.writeBuffer.hasRemaining()) {
                this.writeBuffer = null;
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(new LogEntry("event", "write_response"), new LogEntry("id", this.id), new LogEntry("num_bytes", Integer.toString(numBytes)));
                }
                if (this.httpOneDotZero && !this.keepAlive) {
                    if (ConnectionEventLoop.this.logger.enabled()) {
                        ConnectionEventLoop.this.logger.log(new LogEntry("event", "close_after_response"), new LogEntry("id", this.id));
                    }
                    this.failSafeClose();
                } else if (this.requestParser.parse()) {
                    if (ConnectionEventLoop.this.logger.enabled()) {
                        ConnectionEventLoop.this.logger.log(new LogEntry("event", "pipeline_request"), new LogEntry("id", this.id), new LogEntry("request_bytes", Integer.toString(this.byteTokenizer.remaining())));
                    }
                    this.onParseRequest();
                } else {
                    this.requestTimeoutTask = ConnectionEventLoop.this.timeoutQueue.schedule(this::onRequestTimeout, ConnectionEventLoop.this.options.requestTimeout());
                    this.selectionKey.interestOps(1);
                }
            } else {
                if ((this.selectionKey.interestOps() & 4) == 0) {
                    this.selectionKey.interestOps(4);
                }
                if (ConnectionEventLoop.this.logger.enabled()) {
                    ConnectionEventLoop.this.logger.log(new LogEntry("event", "write"), new LogEntry("id", this.id), new LogEntry("num_bytes", Integer.toString(numBytes)));
                }
            }
        }

        private void failSafeClose() {
            if (this.requestTimeoutTask != null) {
                this.requestTimeoutTask.cancel();
            }
            this.selectionKey.cancel();
            CloseUtils.closeQuietly(this.socketChannel);
        }
    }
}

